/* Faller & Merimaa's precedence model.
 Called from prec_fallermerimaa.m.
 
 ==========================================================
 Last changed:     $Date: 2011-09-13 17:02:31 +0100 (Tue, 13 Sep 2011) $
 Last committed:   $Revision: 285 $
 Last changed by:  $Author: mu31ch $
 ==========================================================
 */

#include "math.h"
#include "mex.h"

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
	/* ====================== DECLARATIONS ====================== */
    
	double *hc_L,*hc_R; /* pointer to hair-cell-modelled fine structure inputs */
    double *frameCount_in; /* pointer to frame count input */
    double *frame_length_in; /* pointer to frame length input */
    double *maxlag_in; /* pointer to maximum lag input */
    double *tau_in,tau; /* (pointer to) cross-correlation exp time constant */
    double *ic_t_in,ic_t; /* (pointer to) interaural coherence threshold */
    double ic; /* interaural coherence */
    
    /* cross- and auto-correlations */
    double *a_LL,*a_RR,*a_LR;
    mxArray *a_LL_mx[1],*a_RR_mx[1],*a_LR_mx[1];
    
    /* cross- and auto-correlations for previous sample */
    double *a_LL_prev,*a_RR_prev,*a_LR_prev; 
    mxArray *a_LL_prev_mx[1],*a_RR_prev_mx[1],*a_LR_prev_mx[1];
    
    /* cross- and auto-correlations for previous sample */
    double *ccg_r; 
    mxArray *ccg_r_mx[1];
    
    /* integrated CCG for current frame and channel */
    double *ccg_ic; 
    mxArray *ccg_ic_mx[1];
    
    double *ccg_out; /* output CCG */
    double *ic_out; /* output interaural coherence */
	int ic_count; /* counter number of samples in frame that exceed IC threshold */
    
    /* sizes */
	mwSize numchans; /* number of frequency channels */
    mwSize frameCount; /* number of frames */
    mwSize frame_length; /* frame length */
    mwSize maxlag; /* maximum lag */
    mwSize numsamples; /* number of samples (time dimension only) */
    mwSize sample; /* matrix sample */
    mwSize lengthCCG; /* length of cross-correlation (2*maxlag)+1 */
    mwSize ccg_r_dims[3]; /* dimensions of running cross-correlation */
	mwSize ccg_dims[3]; /* dimensions of output CCG */
    
    /* indices */
	mwIndex i; /* frequency */
    mwIndex j; /* frame */
    mwIndex n; /* sample */
    mwIndex m; /* lag */
    mwIndex index_ccg_r; /* running cross-correlation */
    mwIndex index_a; /* cross- and auto-correlations */
    mwIndex index_ccg; /* output CCG */
    mwIndex leftn,rightn; /* samples to cross-correlate */
    
	/* ====================== INPUTS ====================== */
	
    /* pointer to hair-cell-modelled fine structure inputs */
	hc_L = mxGetPr(prhs[0]);
	hc_R = mxGetPr(prhs[1]);
    
	frameCount_in = mxGetPr(prhs[2]); /* pointer to frame count input */
	frame_length_in = mxGetPr(prhs[3]); /* pointer to frame length input */
	maxlag_in = mxGetPr(prhs[4]); /* pointer to maximum lag input */
	tau_in = mxGetPr(prhs[5]); /* pointer to cross-correlation exp time constant */
	ic_t_in = mxGetPr(prhs[6]); /* pointer to interaural coherence threshold */
	numsamples = mxGetM(prhs[0]); /* number of input samples (in time dimension only) */
    numchans = mxGetN(prhs[0]); /* number of frequency channels */
	
    /* ====================== DERIVE VARIABLES ====================== */
    
	ic_t = *ic_t_in; /* interaural coherence threshold */
	frameCount = *frameCount_in; /* frame count */
	frame_length = *frame_length_in; /* frame length */
	maxlag = *maxlag_in; /* maximum lag */
	tau = *tau_in; /* cross-correlation exp time constant */
	lengthCCG = (2*maxlag)+1; /* length of cross-correlation */
    
    /* dimensions of running cross-correlation */
    ccg_r_dims[0] = lengthCCG;
	ccg_r_dims[1] = frame_length;
	ccg_r_dims[2] = numchans;
    
    /* dimensions of output CCG */
	ccg_dims[0] = lengthCCG;
	ccg_dims[1] = numchans;
	ccg_dims[2] = frameCount;
	
	/* cross- and auto-correlations */
	a_LL_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
	a_RR_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
	a_LR_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
    a_LL = mxGetPr(a_LL_mx[0]);
	a_RR = mxGetPr(a_RR_mx[0]);
	a_LR = mxGetPr(a_LR_mx[0]);
    
    /* cross- and auto-correlations for previous sample */
	a_LL_prev_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
	a_RR_prev_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
	a_LR_prev_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
    a_LL_prev = mxGetPr(a_LL_prev_mx[0]);
	a_RR_prev = mxGetPr(a_RR_prev_mx[0]);
	a_LR_prev = mxGetPr(a_LR_prev_mx[0]);
    
    /* cross- and auto-correlations for previous sample */
    ccg_r_mx[0] = mxCreateNumericArray((mwSize)3,ccg_r_dims,mxDOUBLE_CLASS,mxREAL);
    ccg_r = mxGetPr(ccg_r_mx[0]);
    
    /* integrated CCG for current frame and channel */
	ccg_ic_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)1,mxREAL);
	ccg_ic = mxGetPr(ccg_ic_mx[0]);
	
	/* ====================== OUTPUTS ====================== */
    
    /* CCG */
	plhs[0] = mxCreateNumericArray(3,ccg_dims,mxDOUBLE_CLASS,mxREAL);
    ccg_out = mxGetPr(plhs[0]);
    
    /* IC */
	plhs[1] = mxCreateDoubleMatrix((mwSize)numsamples,(mwSize)numchans,mxREAL);
    ic_out = mxGetPr(plhs[1]);
	
	/* ====================== CROSS-CORRELATE ====================== */
    
	for ( j = 0; j < frameCount; j++ ) {
		for ( i = 0; i < numchans; i++ ) {
			index_ccg = (i+j*numchans)*lengthCCG;
			sample = (i*(mwIndex)numsamples)+(j*(mwIndex)frame_length);
			for ( m = 0; m < lengthCCG; m++ ) {/* reset ccg for T-F unit */
				ccg_ic[m] = 0.0;
			}
			ic_count = 0; /* reset the count of samples with IC values over the IC threshold */
			for ( n = 0; n < frame_length; n++ ) {
				index_a = (i*lengthCCG);
				index_ccg_r = (n+i*frame_length)*lengthCCG;
				ic = 0.0; /* reset the IC value for the sample */
				for ( m = 0; m < lengthCCG; m++ ) { /* calculate cross-correlations */
					leftn = sample+n+( (m>maxlag) ? (m-maxlag) : (0) );
                    rightn = sample+n+( (m<maxlag) ? (maxlag-m) : (0) );
                    a_LR[index_a+m] = ((1.0/tau)*hc_L[leftn]*hc_R[rightn]) + ((1.0-(1.0/tau))*a_LR_prev[index_a+m]);
					a_LL[index_a+m] = ((1.0/tau)*hc_L[leftn]*hc_L[leftn]) + ((1.0-(1.0/tau))*a_LL_prev[index_a+m]);
					a_RR[index_a+m] = ((1.0/tau)*hc_R[rightn]*hc_R[rightn]) + ((1.0-(1.0/tau))*a_RR_prev[index_a+m]);
					ccg_r[index_ccg_r+m] = a_LR[index_a+m]/sqrt(a_LL[index_a+m]*a_RR[index_a+m]);
					if ( (a_LL[index_a+m]*a_RR[index_a+m])==0 ) { /* prevent divide by zero */
						ccg_r[index_ccg_r+m] = 0.0;
					}
					ic = ( (ic>ccg_r[index_ccg_r+m]) ? (ic) : (ccg_r[index_ccg_r+m]) ); /* calculate IC for sample */
					a_LR_prev[index_a+m] = a_LR[index_a+m];
					a_LL_prev[index_a+m] = a_LL[index_a+m];
					a_RR_prev[index_a+m] = a_RR[index_a+m];
				}
				ic_out[sample+n] = ic;
				if ( ic >= ic_t ) { /* check if IC exceeds IC threshold */
					ic_count += 1; /* count the number of samples that exceed the IC threshold */
					for ( m = 0; m < lengthCCG; m++ ) { /* add ccg with high enough IC */
						ccg_ic[m] += ccg_r[index_ccg_r+m];
					}
				}
			}
			if ( ic_count == 0 ) {
				for ( m = 0; m < lengthCCG; m++ ) { /* write zeros to output if no samples in frame have high enough IC */
					ccg_out[index_ccg+m] = 0.0;
				}
			}
			else {
				for ( m = 0; m < lengthCCG; m++ ) { /* average ccgs with high enough IC */
					ccg_out[index_ccg+m] = ccg_ic[m]/(double)ic_count;
				}
			}				
		} /* end frequency loop */
	}
    /* Destroy mx arrays */
	mxDestroyArray(*ccg_r_mx);
	mxDestroyArray(*ccg_ic_mx);
	mxDestroyArray(*a_LL_mx);
	mxDestroyArray(*a_RR_mx);
	mxDestroyArray(*a_LR_mx);
	mxDestroyArray(*a_LL_prev_mx);
	mxDestroyArray(*a_RR_prev_mx);
	mxDestroyArray(*a_LR_prev_mx);
	return;
}
