/* Lindemann's precedence model.
 Called from prec_lindemann.m.
 
 ==========================================================
 Last changed:     $Date: 2011-09-13 17:02:31 +0100 (Tue, 13 Sep 2011) $
 Last committed:   $Revision: 285 $
 Last changed by:  $Author: mu31ch $
 ==========================================================
 */

#include "math.h"
#include "mex.h"

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
	/* ====================== DECLARATIONS ====================== */
    
	double *hc_L,*hc_R; /* pointer to hair-cell-modelled fine structure inputs */
    double *frameCount_in; /* pointer to frame count input */
    double *frame_length_in; /* pointer to frame length input */
    double *maxlag_in; /* pointer to maximum lag input */
    double *fs_in; /* pointer to sampling frequency */
    double fs; /* sampling frequency */
    double alpha,alpha2; /* fade-off and integration time constant */
    double *tinh_in; /* pointer to fade-off time constant */
    double tinh; /* fade-off time constant */
    double td,tint; /* half sample period and integration time constant */
    double Mf; /* fading constant for monaural sensitivity */
    double wf; /* monaural sensitivity of a correlator to the signal at the end of a delay line */
    
    /* monaural sensitivity functions */
	double *wl,*wr; 
    mxArray *wl_mx[1],*wr_mx[1];
    
    /* running cross-correlation */
    double *k;
    mxArray *k_mx[1];
    
    /* input to cross-correlation */
    double *lin,*rin;
    mxArray *lin_mx[1],*rin_mx[1];
    
    /* inhibited input to the cross-correlation */
    double *lc,*rc;
    mxArray *lc_mx[1],*rc_mx[1];
    
    /* inhibitory components */
    double *il,*ir;
    mxArray *il_mx[1],*ir_mx[1];
    
    /* dynamic inhibitory component for current and previous samples respectively */
    double *phin,*phic;
    mxArray *phin_mx[1],*phic_mx[1];
    
    /* inhibited cross-correlation for current and previous samples respectively */
    double *sumn,*sumc;
    mxArray *sumn_mx[1],*sumc_mx[1];
    
    double *ccg_out; /* output CCG */
	
    /* sizes */
	mwSize numchans; /* number of frequency channels */
    mwSize frameCount; /* number of frames */
    mwSize frame_length; /* frame length */
    mwSize maxlag; /* maximum lag */
    mwSize numsamples; /* number of samples (time dimension only) */
    mwSize sample; /* matrix sample */
    mwSize lengthCCG; /* length of cross-correlation (2*maxlag)+1 */
    mwSize in_length; /* length of input to cross-correlation */
    mwSize n_sample; /* length of inhibited input to cross-correlation */
	mwSize ccg_dims[3]; /* dimensions of output CCG */
    
    /* indices */
	mwIndex i; /* frequency */
    mwIndex j; /* frame */
    mwIndex n; /* sample */
    mwIndex m; /* lag */
    mwIndex index_n; /* frequency */
    mwIndex index_ccg; /* output CCG */
    mwIndex leftn,rightn; /* left and right samples to be cross-correlated */
	
	/* ====================== INPUTS ====================== */
	
	/* pointers to hair-cell-modelled fine structure inputs */
	hc_L = mxGetPr(prhs[0]);
	hc_R = mxGetPr(prhs[1]);
    
	frameCount_in = mxGetPr(prhs[2]); /* pointer to frame count input */
	frame_length_in = mxGetPr(prhs[3]); /* pointer to frame length input */
	maxlag_in = mxGetPr(prhs[4]); /* pointer to maximum lag input */
	fs_in = mxGetPr(prhs[5]); /* pointer to sampling frequency input */
	tinh_in = mxGetPr(prhs[6]); /* pointer to fade-off time constant */
	numsamples = mxGetM(prhs[0]); /* number of input samples (in time dimension only) */
    numchans = mxGetN(prhs[0]); /* number of frequency channels */
	
    /* ====================== DERIVE VARIABLES ====================== */
	
	frameCount = *frameCount_in; /* frame count input */
	frame_length = *frame_length_in; /* frame length input */
	maxlag = *maxlag_in; /* maximum lag */
	lengthCCG = (2*maxlag)+1; /* length of cross-correlation */
	fs = *fs_in; /* sampling frequency */
	tinh = *tinh_in/1000.0; /* fade-off time constant */
	td = 0.5*(1.0/fs); /* half sample period */
	tint = 0.005; /* integration time constant */
	in_length = lengthCCG+frame_length-1; /* length of input to cross-correlation */
	n_sample = 2*in_length-1; /* length of inhibited input to cross-correlation */
	Mf = 6.0; /* fading constant for monaural sensitivity */
	wf = 0.035; /* monaural sensitivity of a correlator to the signal at the end of a delay line */
	alpha = exp(-td/tinh); /* fade-off time constant */
	alpha2 = exp(-td/tint); /* integration time constant */
    
	/* dimensions of output CCG */
	ccg_dims[0] = lengthCCG;
	ccg_dims[1] = numchans;
	ccg_dims[2] = frameCount;
	
    /* monaural sensitivity functions */
	wl_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)1,mxREAL);
	wr_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)1,mxREAL);
    wl = mxGetPr(wl_mx[0]);
	wr = mxGetPr(wr_mx[0]);
	for ( m = 0; m < lengthCCG; m++ ) {
		wl[m] = wf*exp(-((double)m)/Mf);
		wr[m] = wf*exp(((double)m-(double)lengthCCG+1.0)/Mf);
	}
    
    /* running cross-correlation */
	k_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)1,mxREAL);
    k = mxGetPr(k_mx[0]);
    
    /* inhibited input to the cross-correlation */
	lc_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
	rc_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
    lc = mxGetPr(lc_mx[0]);
	rc = mxGetPr(rc_mx[0]);
    
    /* input to cross-correlation */
	lin_mx[0] = mxCreateDoubleMatrix((mwSize)n_sample,(mwSize)1,mxREAL);
	rin_mx[0] = mxCreateDoubleMatrix((mwSize)n_sample,(mwSize)1,mxREAL);
	lin = mxGetPr(lin_mx[0]);
	rin = mxGetPr(rin_mx[0]);
    
    /* inhibitory components */
	il_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
	ir_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
    il = mxGetPr(il_mx[0]);
	ir = mxGetPr(ir_mx[0]);
    
    /* dynamic inhibitory component for current and previous samples */
	phin_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)1,mxREAL);
	phic_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
    phin = mxGetPr(phin_mx[0]);
	phic = mxGetPr(phic_mx[0]);
    
    /* inhibited cross-correlation for current and previous samples */
	sumn_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)1,mxREAL);
	sumc_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
    sumn = mxGetPr(sumn_mx[0]);
	sumc = mxGetPr(sumc_mx[0]);
	
	/* ====================== OUTPUT ====================== */
    
	plhs[0] = mxCreateNumericArray(3,ccg_dims,mxDOUBLE_CLASS,mxREAL);
	ccg_out = mxGetPr(plhs[0]);
	
	/* ====================== CROSS-CORRELATE ====================== */
	
	for ( j = 0; j < frameCount; j++ ) {
		for ( i = 0; i < numchans; i++ ) {
			index_ccg = (i+j*numchans)*lengthCCG;
			sample = (i*(mwIndex)numsamples)+(j*(mwIndex)frame_length);
			for ( n = 0 ; n < in_length-1 ; n++ ) {
                /* input to cross-correlation */
				lin[(2*n)] = hc_L[sample+n];
				rin[(2*n)] = hc_R[sample+n];
			}
			for ( n = 0; n < n_sample; n++ ) {
				index_n = (i*lengthCCG);
				for ( m = 0; m < 2*maxlag; m++ ) {
                    /* inhibit input to cross-correlation */
					lc[index_n+m] = lc[index_n+m+1]*il[index_n+m+1];
					rc[index_n+(2*maxlag)-m] = rc[index_n+(2*maxlag)-m-1]*ir[index_n+(2*maxlag)-m-1];
				}
				lc[index_n+lengthCCG-1] = lin[n];
				rc[index_n] = rin[n];
                /* cross-correlate */
				for ( m = 0; m < lengthCCG; m++ ) {
					k[m] = (wl[m]+(1.0-wl[m])*rc[index_n+m])*(wr[m]+(1.0-wr[m])*lc[index_n+m]);
					phin[m] = k[m]+alpha*phic[index_n+m]*(1-k[m]);
					phic[index_n+m]=phin[m];
					il[index_n+m]=(1.0-rc[index_n+m])*(1-phic[index_n+m]);
					ir[index_n+m]=(1.0-lc[index_n+m])*(1-phic[index_n+m]);
					sumn[m]=sumc[index_n+m]*alpha2+(1.0-alpha2)*k[m];
					sumc[index_n+m]=sumn[m];
					ccg_out[index_ccg+m] += sumn[m]/(double)n_sample;
				}
			}
		} /* end frequency loop */
	}/* end frame loop */
    /* Destroy mx arrays */
	mxDestroyArray(*wl_mx);
	mxDestroyArray(*wr_mx);
	mxDestroyArray(*k_mx);
	mxDestroyArray(*lc_mx);
	mxDestroyArray(*rc_mx);
	mxDestroyArray(*il_mx);
	mxDestroyArray(*ir_mx);
	mxDestroyArray(*phin_mx);
	mxDestroyArray(*phic_mx);
	mxDestroyArray(*sumn_mx);
	mxDestroyArray(*sumc_mx);
	mxDestroyArray(*lin_mx);
	mxDestroyArray(*rin_mx);
	return;
}
