function [cont,dirflag] = get_contents(directory,str)
%  Get the contents of a specified directory
%
%   [cont,dirflag] = get_contents(directory,str)
%
%   This function returns the contents of a specified
%   directory.
%
%   cont = get_contents(directory) obtains the contents
%   (folder and file names) of directory and returns them to
%   the cell array cont. It ignores hidden files and folders
%   (those starting '.').
% 
%   cont = get_contents(directory,str) allows an additional
%   filter string str to be specified. Only elements that
%   contain str will be returned (files or folders. In
%   addition, str may be specified as one of three keywords:
%   'files' (to return names of all files in the directory),
%   'folders' (to return names of all folders in the
%   directory), or '*.ext' where 'ext' is a user-specified
%   file extension (to return all files with the extension
%   '.ext'). str is case sensitive.
% 
%   [cont,dirflag] = get_contents(...) returns a logical
%   array dirflag, the same size as cont, indicating whether
%   each element is a directory.
%
%   Both inputs must be strings, specifying a single folder
%   and single filter term.
% 
%   Example
% 
%       % Return all m-files in the current directory:
%       cont = get_contents(cd,'*.m')

% !---
% ==========================================================
% Last changed:     $Date: 2011-09-13 17:01:41 +0100 (Tue, 13 Sep 2011) $
% Last committed:   $Revision: 71 $
% Last changed by:  $Author: mu31ch $
% ==========================================================
% !---

cont = struct2cell(dir(directory));
dirbool = cell2mat(cont(cellfun(@islogical,cont(:,1)),:)); % return directory flags
cont = cont(1,:); % keep only file names
X = ~strncmp(cont, '.', 1); % remove hidden files (those starting '.')
cont = cont(X);
cont = cont(:); % make column vector
dirbool = dirbool(X);
dirbool = dirbool(:); % make column vector

if nargin > 1
    % find filename extensions
    exts = cell(size(cont));
    for n = 1:length(cont)
        [~,~,exts{n}] = fileparts(cont{n});
    end
    % filter
    if strncmp(str,'*.',2) % if extensions are requested
        ext = str(2:end);
        str = 'ext';
    end
    switch lower(str)
        case 'files'
            Y = ~dirbool;
        case 'folders'
            Y = dirbool;
        case 'ext'
            Y = strcmp(exts,ext);
        otherwise % use literal search string
            Y = ~cellfun(@isempty,strfind(cont,str));
    end
else
    Y = true(size(cont));
end

cont = cont(Y);
if nargout > 1    
    dirflag = dirbool(Y);
end

% [EOF]
