/* Convert cross-correlograms to azimuth-warp cross-correlograms and skeleton cross-correlograms.
 Called from warp_ccg.m.
 
 ==========================================================
 Last changed:     $Date: 2011-09-13 17:02:31 +0100 (Tue, 13 Sep 2011) $
 Last committed:   $Revision: 285 $
 Last changed by:  $Author: mu31ch $
 ==========================================================
 */

#include "math.h"
#include "mex.h"

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
    /* ====================== DECLARATIONS ====================== */
	
	double *ccg_in; /* pointer to input cross-correlograms */
    double *sd; /* pointer to input standard deviations */
    double *warpers_in; /* pointer to input azimuth-warping function */
    
    /* pointer to gaussian functions */
    double *gaussians; 
    mxArray *gaussians_mx[1];
    
    /* pointer to inputs of warping interpolator */
    double *warp_in_ccg,*warp_in_warper,*warp_in_tau;
    mxArray *warp_in[4];
    
    /* pointer to output of warping interpolator */
    double *warp_out_ccg;
    mxArray *warp_out[1];
    
    /* pointer to inputs of 'skeletonising' convolution */
    double *conv_in_data,*conv_in_gauss;
    mxArray *conv_in[2];
    
    /* pointer to output of convolution */
    double *conv_out_data;
    mxArray *conv_out[1];
    
    double *ccg_azi_out; /* pointer to function output: azimuth-warped cross-correlogram */
    double *sccg_out; /* pointer to function output: azimuth-warped skeleton cross-correlogram */
	
    /* Indices */
	mwIndex i; /* frequency */
    mwIndex j; /* frame */
    mwIndex m; /* lag */
    mwIndex index_ccg; /* cross-correlogram */
    mwIndex index_sccg; /* skeleton cross-correlogram */
    mwIndex lengthCCG; /* number of lags (2*maxlag)+1 */
    mwIndex newLengthCCG; /* number of angles (2*maxdeg)+1 */
	
    /* Dimensions */
	mwSize sccg_dims[3]; /* skeleton cross-correlogram */
    mwSize numchans; /* number of frequency channels */
    mwSize frameCount; /* number of frames */
    mwSize maxlag; /* maximum lag */
    mwSize maxdeg; /* maximum angle */
    const mwSize *dims; /* dimensions of input cross-correlogram */
	
	/* ====================== INPUTS ====================== */
    
	ccg_in = mxGetPr(prhs[0]); /* input cross-correlograms */
    sd = mxGetPr(prhs[1]); /* input standard deviations */
    warpers_in = mxGetPr(prhs[2]); /* input azimuth-warping function */
	
	/* ====================== DERIVE VARIABLES ====================== */
    
    dims = mxGetDimensions(prhs[0]); /* dimensions of input cross-correlogram */
    lengthCCG = (mwIndex)dims[0]; /* number of lags (2*maxlag)+1 */
    numchans = dims[1]; /* number of frequency channels */
    frameCount = dims[2]; /* number of frames */
    maxlag = ((int)lengthCCG-1)/2; /* maximum lag */
    
	newLengthCCG = (mwIndex)mxGetM(prhs[2]); /* number of angles (2*maxdeg)+1 */
	maxdeg = ((int)newLengthCCG-1)/2; /* maximum angle */
	
    /* Output dims for azimuth-warped cross-correlogram and azimuth-warped skeleton cross-correlogram */
    sccg_dims[0] = (mwSize)newLengthCCG;
	sccg_dims[1] = numchans;
	sccg_dims[2] = frameCount;
    
    /* ====================== OUTPUTS ====================== */
    
    /* function output: azimuth-warped cross-correlogram */
	plhs[0] = mxCreateNumericArray(3,sccg_dims,mxDOUBLE_CLASS,mxREAL);
    ccg_azi_out = mxGetPr(plhs[0]);
    
    /* function output: azimuth-warped skeleton cross-correlogram */
    plhs[1] = mxCreateNumericArray(3,sccg_dims,mxDOUBLE_CLASS,mxREAL);
	sccg_out = mxGetPr(plhs[1]);
    
    /* ====================== MATRICES ====================== */
	
	/* initiate reference for warping */
	warp_in[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)1,mxREAL);
	warp_in_tau = mxGetPr(warp_in[0]);
	
	for ( m = 0; m < (int)lengthCCG; m++ ) {
		warp_in_tau[m] = (((double)m-(double)maxlag)/(double)maxlag);
	}
    
	/* initiate ccg input to warping */
	warp_in[1] = mxCreateDoubleMatrix((mwSize)lengthCCG,1,mxREAL);
	warp_in_ccg = mxGetPr(warp_in[1]);
    
	/* initiate warping function array */
	warp_in[2] = mxCreateDoubleMatrix((mwSize)newLengthCCG,1,mxREAL);
	warp_in_warper = mxGetPr(warp_in[2]);
    
    /* interpolation type for warping */
	warp_in[3] = mxCreateString("spline");
	
	/* initiate data for skeleton ccg convolution */
	conv_in[0] = mxCreateDoubleMatrix((mwSize)newLengthCCG,(mwSize)1,mxREAL);
    conv_in_data = mxGetPr(conv_in[0]);
    
    /* initiate gaussians for skeleton ccg convolution */
	conv_in[1] = mxCreateDoubleMatrix((mwSize)newLengthCCG,(mwSize)1,mxREAL);
	conv_in_gauss = mxGetPr(conv_in[1]);
	
    /* derive gaussian functions */
	gaussians_mx[0] = mxCreateDoubleMatrix((mwSize)newLengthCCG,(mwSize)numchans,mxREAL);
	gaussians = mxGetPr(gaussians_mx[0]);
	for ( i = 0; i < numchans; i++ ) {
		for ( m = 0; m < (int)newLengthCCG; m++ ) {
			gaussians[(i*newLengthCCG)+m] = exp((-pow((double)m-(double)maxdeg,2))/(2*(pow(sd[i],2))));
		}
	}
    
	/* ====================== WARP AND SKELETONISE ====================== */
    
	for ( j = 0; j < frameCount; j++ ) {
		for ( i = 0; i < numchans; i++ ) {
            index_ccg = (i*lengthCCG)+(j*lengthCCG*numchans);
			index_sccg = (i*newLengthCCG)+(j*newLengthCCG*numchans);
			/* Do warping stuff */
			for ( m = 0; m < (int)lengthCCG; m++ ) { /* get data to warp */
				warp_in_ccg[m] = ccg_in[index_ccg+m];
			}
			for ( m = 0; m < (int)newLengthCCG; m++ ) { /* get warping function */
				warp_in_warper[m] = warpers_in[(i*newLengthCCG)+m];
			}
			mexCallMATLAB(1,warp_out,4,warp_in,"interp1"); /* interpolate */
			warp_out_ccg = mxGetPr(warp_out[0]); /* get output of interpolator */
			for ( m = 0; m < (int)newLengthCCG; m++ ) {
				ccg_azi_out[index_sccg+m] = warp_out_ccg[m]; /* write azimuth-warped CCG output */
			}
			/* Skeleton ccg */
			for ( m = 1; m < (int)newLengthCCG-1; m++ ) {
				if ( warp_out_ccg[m] > warp_out_ccg[m-1] && warp_out_ccg[m] > warp_out_ccg[m+1] ) { /* extract peaks */
					conv_in_data[m] = warp_out_ccg[m];
				}
				else {
					conv_in_data[m] = 0.0; /* non-peaks to zero */
				}
			}
			for ( m = 0; m < (int)newLengthCCG; m++ ) { /* get gaussian for channel */
				conv_in_gauss[m] = gaussians[(i*newLengthCCG)+m];
			}
			mexCallMATLAB(1,conv_out,2,conv_in,"conv_fft"); /* convolve */
			conv_out_data = mxGetPr(conv_out[0]); /* get output of convolver */
			for ( m = 0; m < (int)newLengthCCG; m++ ) {
				sccg_out[index_sccg+m] = conv_out_data[m+(mwIndex)maxdeg]; /* write azimuth-warped skeleton CCG output */
			}
			mxDestroyArray(warp_out[0]);
			mxDestroyArray(conv_out[0]);
		} /* end frequency loop */
	} /* end frame loop */
    /* Destroy mx arrays */
	mxDestroyArray(*warp_in);
    mxDestroyArray(*conv_in);
    mxDestroyArray(*gaussians_mx);
	return;
}
