function [svn_profile,svn_str] = build_svn_profile(folders,keywords,strs)
% Read data from files tagged with SVN keywords
% 
%   [svn_profile,svn_str] = ...
%       build_svn_profile(folders,keywords,strs)
% 
% 
%   This function extracts SVN keyword data from specified
%   files, and returns the filename, path and keyword data
%   to a cell array (and optional char array).
% 
%   svn_profile = build_svn_profile(folders,keywords)
%   returns a cell array svn_profile containing data
%   associated with the SVN keywords. The function will
%   search through files contained in folders.
% 
%   svn_profile is a two dimensional cell array; there is
%   one row for each file found in folders, and N columns:
%   one each for filename and path, and appended columns for
%   each specified keyword. Data is only returned for files
%   that contain the specified keyword(s). An empty cell
%   array, 0-by-3, is returned if no keywords are found.
% 
%   svn_profile = build_svn_profile(folders,keywords,strs)
%   allows additional filter strings strs to be specified.
%   See GET_CONTENTS for details of permitted filter
%   strings.
% 
%   [svn_profile,svn_str] = build_svn_profile(...) outputs a
%   char array svn_str of the profile, with columns
%   concatenated to produce continuous lines. Spaces are
%   inserted to ensure that columns are aligned (in a
%   monospaced font). The array is appropriate for printing
%   or embedding in a PostScript file, for example.
% 
%   folders, keywords,and strs may be a char array
%   specifying a single occurrence, or a cell array of
%   strings specifying multiple occurrences.
% 
%   Keyword data are placed in files automatically by
%   Subversion at each commit using the 'keyword' function.
%   The data can be useful in maintaining an audit trail, or
%   establishing the version used to generate a particular
%   set of results.
% 
%   See also READ_SVN_KEYWORD, GET_CONTENTS,
%   ADD_PS_COMMENTS.

% !---
% ==========================================================
% Last changed:     $Date: 2011-09-13 17:01:41 +0100 (Tue, 13 Sep 2011) $
% Last committed:   $Revision: 71 $
% Last changed by:  $Author: mu31ch $
% ==========================================================
% !---

if ischar(folders)
    folders = cellstr(folders);
end

if ischar(keywords)
    keywords = cellstr(keywords);
end

if nargin > 2
    if ischar(strs)
        strs = cellstr(strs);
    end
else
    strs = {'files'};
end

keydata = cell(0,2+length(keywords));
n = 1;
for f = 1:length(folders)
    folder = folders{f};
    for s = 1:length(strs)
        str = strs{s};
        [files,dirflag] = get_contents(folder,str);
        files = files(~dirflag);
        if ~isempty(files)
            for h = 1:length(files)
                file = files{h};
                for k = 1:length(keywords)
                    keyword = keywords{k};
                    keydata{n,1} = file;
                    keydata{n,2} = [folder filesep];
                    keydata{n,2+k} = read_svn_keyword([folder filesep file],keyword);
                end
                n = n+1;
            end
        end
    end
end

 % remove files where no keyword is found
IX = true(size(keydata(:,3)));
for k = 1:length(keywords)
    IX = IX & cellfun(@isempty,keydata(:,2+k));
end
svn_profile = keydata(~IX,:);

% Build char array version of data
space = repmat('  ',size(svn_profile(:,1))); % spaces to insert between columns
svn_str = [char(svn_profile(:,1)) space char(svn_profile(:,2))]; % first two columns (filename, path)
% append keyword columns:
for k = 1:length(keywords)
        svn_str = [svn_str space char(svn_profile(:,2+k))]; %#ok<AGROW>
end

% [EOF]
